/*
 * Copyright: 2018 WERK II GmbH, Germany [www.priint.com].
 * 
 * WERK II permits you to use this file in accordance with the terms of the WERK II license agreement
 * accompanying it. If you have received this file from a source other than WERK II, then your use
 * of it requires the prior written permission of WERK II.
 *  
 * Contact info@priint.com if you have any questions regarding the license.
 * 
 */
package com.priint.pubserver.auth.realm;

import java.net.URL;
import java.util.Set;
import java.util.UUID;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.Unmarshaller;

import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authc.AuthenticationInfo;
import org.apache.shiro.authc.AuthenticationToken;
import org.apache.shiro.authc.SimpleAuthenticationInfo;
import org.apache.shiro.authc.UsernamePasswordToken;
import org.apache.shiro.authz.AuthorizationInfo;
import org.apache.shiro.realm.AuthorizingRealm;
import org.apache.shiro.subject.PrincipalCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.priint.pubserver.auth.realm.DemoUsers.DemoUser;
import com.werkii.server.ejb.admin.AdminLocal;
import com.werkii.server.entities.Serveruser;
import com.werkii.server.utils.CometServer4ServiceLocator;

/**
 * Demo realm.
 * <p>
 * We just read some users with email, password,  and group memberships from disk or URL.
 * <p>
 * On a login event, we first validate the user and then we synchronize the user into pubserver.
 * <p>
 * The main method (doGetAuthenticationInfo) ends by returning a SimpleAuthenticationInfo to indicate successful processing of the login
 * request to the Shiro processing chain.
 * 
 *
 */
public class DemoRealm extends AuthorizingRealm {

	private static final Logger logger = LoggerFactory.getLogger(DemoRealm.class);

	private DemoConfig config = new DemoConfig();
	private DemoUsers demoUsers;
	
	@Override
	protected AuthorizationInfo doGetAuthorizationInfo(PrincipalCollection arg0) {
		return null;
	}

	@Override
	protected AuthenticationInfo doGetAuthenticationInfo(AuthenticationToken token) {

		if (!(token instanceof UsernamePasswordToken)) {
			// if token type is not supported die early 
			return null;
		}
		
		// read credentials from token
		UsernamePasswordToken usernamePasswordToken = (UsernamePasswordToken) token;
		String loginname = usernamePasswordToken.getUsername();
		String password = new String(usernamePasswordToken.getPassword());

		// check if provided user is excluded nor not included by rule stated in config
		if (loginname == null 
				|| (config.getRegexpExclude() != null && config.getRegexpExclude().matcher(loginname).matches())
				|| (config.getRegexpInclude() != null && !config.getRegexpInclude().matcher(loginname).matches())) {
			return null;
		}
		
		// Ensure that configured users are loaded into demoUsers field. 
		// Here we just read from a cached document.		
		// In a real realm user lookup and validation will use a life connection
		// e.g. e.g. a database or restful request.  
		readUsers();
		
		// identify the user - may be null
		DemoUser user = findUser(loginname);
		
		// for the sake of simplicity we use our own credential matching here
		if (!validateCredentials(user, password)) {
			return null;
		}
		
		// So far everything was shiro standard procedure. 
		// Now comes the pubserver specific part.
		// We synchronize the user and the group memberships of the DemoUser into PubServer using the mappings as defined in the config.
		synchronizeUser(user);
	
		return new SimpleAuthenticationInfo(user.getUsername(), password, getName());		

	}

	/**
	 * Synchronize the user and the group memberships of the DemoUser into PubServer using the mappings as defined in the config.
	 * @param demoUser
	 */
	private void synchronizeUser(DemoUser demoUser) {
		try {
			
			// create the pubserver user object that will be synced
			Serveruser serverUser = createServerUser(demoUser);

			// retrieve the pubserver role names related to the groups of the user
			Set<String> roleNames = config.mapMembershipsToRoles(demoUser.getGroups());
			
			// for synchronization we need a reference to the publishing planner admin interface 
			AdminLocal adminBean = CometServer4ServiceLocator.ServiceLocatorEnum.INSTANCE.getAdminLocal();

			// run the sync
			// this is actually a kind of upsert process for the user object (update or create)
			// deprecated roles will be removed from the memberships and new ones will be added
			// after login user fields can still be edited via planner Web UI but changes will be lost during a new login 
			adminBean.synchronizeUser(serverUser, roleNames, config.getDefaultDataset(), config.getDefaultRole());
			
		} catch (Exception e) {
			logger.warn("User synchronization process failed for user={} and groups={}. {}", 
					demoUser.getUsername(), demoUser.getGroups(), e.getMessage());
			throw new AuthenticationException("Error synchronizing DemoUser '" + demoUser.getUsername() + "' into pubserver", e);
		}
	}

	/**
	 * Find a user in the list of active users.
	 * <p>
	 * We expect that users login by email address.
	 * 
	 * @param loginname
	 * @return A DemoUser or null.
	 */
	private DemoUser findUser(String loginname) {
		if (loginname != null && !loginname.isEmpty() &&  demoUsers != null) {
			for (DemoUser user : demoUsers.getUsers()) {
				if (loginname.equals(user.getEmail())) {
					return user;
				}
			}
		}
		return null;
	}

	/**
	 * For real realm this will be a little bit more sophisticated ;-)
	 * @param user
	 * @param password
	 * @return
	 */
	private boolean validateCredentials(DemoUser user, String password) {
		return user!=null && password!=null && !password.isEmpty() && password.equals(user.getPassword());
	}

	/**
	 * Read configuration from URL and initialize the list of active users if not already done.
	 * <p>
	 * In case of errors the user list will be empty and user login via this realm will not be possible.
	 */
	private void readUsers() {
		if (demoUsers == null) {
			try {
				JAXBContext jaxbContext  = JAXBContext.newInstance(DemoUsers.class);
				Unmarshaller jaxbUnmarshaller = jaxbContext.createUnmarshaller();
				demoUsers = (DemoUsers) jaxbUnmarshaller.unmarshal(new URL(config.getUrl()));
			} catch (Exception e) {
				// We just log the error.
				// Realm will reject all login attempts because userlist is empty. 
				logger.error("Bad realm configuration! URL '{}' could not be read from or does not contain a valid DemoUsers objects.", config.getUrl(), e);
			}
		}		
	}

	
	/**
	 * Creating a pubserver {@link Serveruser} object for user synchronization.
	 * <p> Fields correspond to WebUI in publishing planner user administration.  
	 * @param demoUser
	 * @return
	 */
	private Serveruser createServerUser(DemoUser demoUser) {
		
		Serveruser serverUser = new Serveruser();

		// we only fill some name fields and set the other one to some empty or default value
		serverUser.setLogin(demoUser.getUsername());
		serverUser.setGivenname(demoUser.getUsername());
		serverUser.setName(demoUser.getUsername());

		serverUser.setSurname("-");
		serverUser.setEmail(demoUser.getEmail());
		serverUser.setColorId(null);
		serverUser.setCreatedbyid(config.getName()); 
		serverUser.setUpdatedbyid(config.getName());
		serverUser.setPersonId(" ");
		serverUser.setDefaultMenuID(StringUtils.defaultString(config.getDefaultMenu(), " "));
		
		// password field exists for compatibility reasons for priint:suite 3
		// you should fill this by a random value
		serverUser.setPassword(UUID.randomUUID().toString());
		
		return serverUser;
	}
	
}